<?php

namespace App\Telegram\Menus\Admin;

use App\Models\Folder;
use App\Telegram\Menus\InlineMenuPaginator;
use Illuminate\Pagination\LengthAwarePaginator;
use SergiX44\Nutgram\Nutgram;
use SergiX44\Nutgram\Telegram\Types\Keyboard\InlineKeyboardButton;

class FoldersMenu extends InlineMenuPaginator
{
    use Helpers;

    protected int $folderId;

    public function start(Nutgram $bot): void
    {
        if (!self::authorizeAdmin($bot, $this)) return;


        $this->menuText("📁 **Folder Management**\n\nSelect an action:", ['parse_mode' => 'Markdown'])
            ->clearButtons()
            ->addButtonRow(InlineKeyboardButton::make('➕ Create Folder', callback_data: 'create@startCreateConversation'))
            ->addButtonRow(InlineKeyboardButton::make('🗑 Delete Folder', callback_data: 'delete@showFoldersToDelete'))
            ->addButtonRow(InlineKeyboardButton::make('🔐 Grant Access', callback_data: 'grant@showFoldersToGrant'))
            ->addButtonRow(InlineKeyboardButton::make('🔒 Revoke Access', callback_data: 'revoke@showFoldersToRevoke'))
            ->addButtonRow(InlineKeyboardButton::make('⬅️ Back', callback_data: 'back@goBack'))
            ->showMenu();
    }

    public function goBack(Nutgram $bot): void
    {
        $this->end();
        AdminMenu::begin($bot);
    }

    public function startCreateConversation(Nutgram $bot): void
    {
        $this->end();
        CreateFolderConversation::begin($bot);
    }

    protected function renderPage(Nutgram $bot, ?int $folderId, int $page, array $data = null): void
    {
        if ($data['title'] === 'Delete Folder') {
            $folders = Folder::orderBy('name')->get();
        } else {
            $folders = Folder::where('parent_id', null)->orderBy('name')->get();
        }

        if ($folders->isEmpty()) {
            $bot->answerCallbackQuery($bot->callbackQuery()->id, 'No folders found.');
            return;
        }

        $this->menuText("**{$data['title']}**\n\nSelect a folder:", ['parse_mode' => 'Markdown'])->clearButtons();


        $paginator = new LengthAwarePaginator($folders->forPage($page, $this->perPage), $folders->count(), $this->perPage, $page);

        foreach ($paginator->items() as $folder) {
            $this->addButtonRow(InlineKeyboardButton::make(
                "📁 {$folder->getPath()}",
                callback_data: "{$folder->id}@{$data['action']}"
            ));
        }

        $this->addControls($paginator, null);
        $this->addButtonRow(InlineKeyboardButton::make('⬅️ Back', callback_data: 'back@start'));

        $this->showMenu();
    }

    public function handlePagination(Nutgram $bot): void
    {
        [$folderId, $page] = explode(':', $bot->callbackQuery()->data);

        $this->renderPage($bot, null, (int)$page, [
            'title' => 'Delete Folder',
            'action' => 'handleDelete'
        ]);
    }

    public function showFoldersToDelete(Nutgram $bot): void
    {
        $this->renderPage($bot, null, 1, [
            'title' => 'Delete Folder',
            'action' => 'handleDelete'
        ]);
    }

    public function handleDelete(Nutgram $bot): void
    {
        $folder = Folder::find($bot->callbackQuery()->data);
        if (!$folder) {
            $bot->answerCallbackQuery($bot->callbackQuery()->id, 'Folder not found!');
            return;
        }
        $folderName = $folder->name;
        $folder->delete();

        $this->menuText("✅ Folder '{$folderName}' deleted successfully!")
            ->clearButtons()
            ->addButtonRow(InlineKeyboardButton::make('⬅️ Back to Folders Menu', callback_data: 'back@start'))
            ->showMenu();
    }

    // Grant Access Flow
    public function showFoldersToGrant(Nutgram $bot): void
    {
        $this->renderPage($bot, null, 1, [
            'title' => 'Grant Access - Select Folder',
            'action' => 'startGrantConversation'
        ]);
    }

    public function startGrantConversation(Nutgram $bot): void
    {
        $this->end();
        // Pass the selected folder_id to the conversation
        GrantAccessConversation::begin(bot: $bot, data: ['folder_id' => $bot->callbackQuery()->data]);
    }

    // Revoke Access Flow
    public function showFoldersToRevoke(Nutgram $bot): void
    {
        $this->renderPage($bot, null, 1, [
            'title' => 'Revoke Access - Select Folder',
            'action' => 'startRevokeConversation'
        ]);
    }

    public function startRevokeConversation(Nutgram $bot): void
    {
        $this->end();
        // Pass the selected folder_id to the conversation
        RevokeAccessConversation::begin(bot: $bot, data: ['folder_id' => $bot->callbackQuery()->data]);
    }

    public function showUsersToRevoke(Nutgram $bot): void
    {
        $this->folderId = (int)$bot->callbackQuery()->data;
        $folder = Folder::with('users')->find($this->folderId);

        if (!$folder) {
            $bot->answerCallbackQuery($bot->callbackQuery()->id, 'Folder not found!');
            return;
        }

        if ($folder->users->isEmpty()) {
            $bot->answerCallbackQuery($bot->callbackQuery()->id, 'No users have access to this folder.');
            return;
        }

        $this->menuText("**Revoke Access from '{$folder->name}'**\n\nSelect a user:", ['parse_mode' => 'Markdown'])->clearButtons();

        foreach ($folder->users as $user) {
            $displayName = $user->first_name . ($user->username ? " (@{$user->username})" : '');
            $this->addButtonRow(InlineKeyboardButton::make(
                "👤 {$displayName}",
                callback_data: "{$user->id}@handleRevokeUser"
            ));
        }
        $this->addButtonRow(InlineKeyboardButton::make('⬅️ Back', callback_data: 'revoke@showFoldersToRevoke'));
        $this->showMenu();
    }

    public function handleRevokeUser(Nutgram $bot): void
    {
        $userId = (int)$bot->callbackQuery()->data;
        $user = \App\Models\User::find($userId);
        $folder = Folder::find($this->folderId);

        if (!$user || !$folder) {
            $bot->answerCallbackQuery($bot->callbackQuery()->id, 'User or folder not found!');
            return;
        }

        $user->folders()->detach($this->folderId);
        $displayName = $user->first_name . ($user->username ? " (@{$user->username})" : '');

        $this->menuText("✅ Access revoked from {$displayName} for folder '{$folder->name}'")
            ->clearButtons()
            ->addButtonRow(InlineKeyboardButton::make('⬅️ Back to Folders Menu', callback_data: 'back@start'))
            ->showMenu();
    }
}
