<?php

namespace App\Telegram\Menus\Admin;

use App\Models\Folder;
use Illuminate\Pagination\LengthAwarePaginator;
use SergiX44\Nutgram\Conversations\Conversation;
use SergiX44\Nutgram\Nutgram;
use SergiX44\Nutgram\Telegram\Types\Keyboard\InlineKeyboardButton;
use SergiX44\Nutgram\Telegram\Types\Keyboard\InlineKeyboardMarkup;

class CreateFolderConversation extends Conversation
{
    private int $perPage = 8;
    use Helpers;

    protected ?string $folderName = null;
    protected ?string $askForFolderMessId = null;


    public function start(Nutgram $bot): void
    {
        if (!self::authorizeAdmin($bot, $this)) return;


        $bot->sendMessage('Please send the name for the new folder:');
        $this->next('askForParent');
    }

    public function askForParent(Nutgram $bot, int $page = 1): void
    {
        $this->renderPage($bot, $page);
    }

    public function createFolder(Nutgram $bot): void
    {
        if (str_ends_with($bot->callbackQuery()->data, 'handlePagination')) {
            $this->handlePagination($bot);
            return;
        }

        if (!$bot->isCallbackQuery() || !str_starts_with($bot->callbackQuery()->data, 'parent:')) {
            $bot->sendMessage('Invalid selection. Please click a button from the list above.');
            $this->askForParent($bot);
            return;
        }

        $bot->answerCallbackQuery();
        $parentId = (int)substr($bot->callbackQuery()->data, 7);
        $parentId = $parentId === 0 ? null : $parentId;

        if ($parentId !== null && !Folder::find($parentId)) {
            $bot->editMessageText('❌ Parent folder not found. Operation cancelled.');
            $this->end();
            FoldersMenu::begin($bot);
            return;
        }

        Folder::create(['name' => $this->folderName, 'parent_id' => $parentId]);

        $bot->editMessageText(
            text: "✅ Folder '{$this->folderName}' created successfully!",
            chat_id: $bot->chatId(),
            message_id: $bot->messageId()
        );

        $this->end();
        FoldersMenu::begin($bot);
    }


    protected function renderPage(Nutgram $bot, int $page): void
    {
        if (!$this->folderName) {
            $name = trim($bot->message()->text ?? '');

            if (empty($name) || Folder::where('name', $name)->exists()) {
                $bot->sendMessage('❌ Folder name cannot be empty and must be unique. Please try again:');
                $this->start($bot);
                return;
            }

            $this->folderName = $name;
        }


        $folders = Folder::on()->orderBy('name')->get();

        $keyboard = InlineKeyboardMarkup::make()->addRow(
            InlineKeyboardButton::make('📁 Root (No parent)', callback_data: 'parent:0')
        );


        $paginator = new LengthAwarePaginator($folders->forPage($page, $this->perPage), $folders->count(), $this->perPage, $page);

        foreach ($paginator->items() as $folder) {
            $keyboard->addRow(InlineKeyboardButton::make("📁 {$folder->getPath()}", callback_data: "parent:{$folder->id}"));
        }

        $this->addConversationControls($paginator, $keyboard);

        if ($this->askForFolderMessId) {
            $bot->editMessageText(
                text: "Select a parent folder for '{$this->folderName}':",
                reply_markup: $keyboard);
        } else {
            $this->askForFolderMessId = $bot->sendMessage(
                text: "Select a parent folder for '{$this->folderName}':",
                reply_markup: $keyboard
            )->message_id;
        }


        $this->next('createFolder');
    }

    public function handlePagination(Nutgram $bot): void
    {
        [$page, $method] = explode('@', $bot->callbackQuery()->data);

        $this->askForParent($bot, (int)$page);
    }
}
