<?php

namespace App\Http\Controllers;

use App\Models\Asset;
use App\Models\Folder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class AssetController extends Controller
{
    public function updateOrder(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'asset_orders' => 'required|array|min:1',
                'asset_orders.*.asset_id' => 'required|integer|exists:assets,id',
                'asset_orders.*.position' => 'required|integer|min:0'
            ]);

            DB::beginTransaction();

            $assetOrders = $request->input('asset_orders', []);

            foreach ($assetOrders as $order) {
                $assetId = $order['asset_id'];
                $position = $order['position'];

                $asset = Asset::find($assetId);
                if ($asset) {
                    $asset->update([
                        'position' => $position
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Asset orders updated successfully'
            ]);

        } catch (ValidationException $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Validation error: ' . $e->getMessage(),
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Error updating asset orders: ' . $e->getMessage()
            ], 500);
        }
    }

    public function confirmDelete(Asset $asset)
    {
        $title = "Are you sure you want to delete the asset <strong>$asset->title</strong>?";
        $desc = 'This action cannot be undone.';
        $deleteRoute = route('admin.assets.destroy', $asset);

        return view('admin.folders.delete', compact( 'title', 'desc', 'deleteRoute'));
    }

    public function destroy(Asset $asset)
    {
        $assetTitle = $asset->title;
        $asset->delete();

        return redirect()->route('admin.folders.index')->with('success', "Asset '{$assetTitle}' deleted successfully!");
    }
}
